"use client"

import { useState } from "react"
import Link from "next/link"

const ProductCard = ({ product }) => {
  const [isHovered, setIsHovered] = useState(false)

  // Fallback product data if not provided
  const {
    id = "1",
    name = "Product Naam",
    slug = "product-naam",
    price = 29.99,
    originalPrice = 39.99,
    image = "/placeholder.svg?key=pgf1t",
    stock = 10,
    category = "Categorie",
    rating = 4.5,
    reviewCount = 42,
    shortDescription = "Korte beschrijving van het product en de werking ervan."
  } = product || {}

  const discountPercentage = originalPrice ? Math.round(((originalPrice - price) / originalPrice) * 100) : 0

  return (
    <div
      className="group relative bg-white rounded-xl overflow-hidden border border-gray-100 transition-all duration-300 hover:shadow-xl hover:border-purple-200"
      onMouseEnter={() => setIsHovered(true)}
      onMouseLeave={() => setIsHovered(false)}
    >
      {/* Ribbon */}
      {discountPercentage > 0 && (
        <div className="absolute top-0 left-0 z-10">
          <div className="bg-gradient-to-r from-red-500 to-pink-500 text-white font-bold py-1 px-4 shadow-md transform rotate-0 flex items-center">
            <svg className="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg">
              <path fillRule="evenodd" d="M10 3a1 1 0 01.707.293l1 1a1 1 0 010 1.414L10.414 7H13a1 1 0 110 2h-3v1h3a1 1 0 110 2h-3v1h3a1 1 0 110 2h-3v2a1 1 0 01-2 0v-2H5a1 1 0 110-2h3v-1H5a1 1 0 010-2h3V9H5a1 1 0 010-2h2.586L6.293 5.707a1 1 0 011.414-1.414l1 1A1 1 0 019 5v8a1 1 0 01-2 0V8.414l-1.293 1.293a1 1 0 01-1.414-1.414l3-3A1 1 0 018 5h.001L9 4.707V5a1 1 0 01-1 1H5a1 1 0 110-2h2.586L6.293 2.707a1 1 0 011.414-1.414l1 1A1 1 0 019 2v1.586L10.293 2.293A1 1 0 0111 2z" clipRule="evenodd" />
            </svg>
            {discountPercentage}% KORTING
          </div>
        </div>
      )}

      {/* Category badge */}
      <div className="absolute top-4 right-4 z-10">
        <span className="bg-purple-100 text-purple-800 text-xs font-medium px-2.5 py-1 rounded-full">
          {category}
        </span>
      </div>

      {/* Stock indicator */}
      <div className="absolute top-4 left-4 z-10">
        {stock > 0 ? (
          <div className="flex items-center">
            <span className="h-2.5 w-2.5 bg-green-500 rounded-full mr-1.5"></span>
            <span className="text-xs font-medium text-gray-700 bg-white/80 backdrop-blur-sm px-2 py-0.5 rounded-full">
              Op voorraad
            </span>
          </div>
        ) : (
          <div className="flex items-center">
            <span className="h-2.5 w-2.5 bg-red-500 rounded-full mr-1.5"></span>
            <span className="text-xs font-medium text-gray-700 bg-white/80 backdrop-blur-sm px-2 py-0.5 rounded-full">
              Uitverkocht
            </span>
          </div>
        )}
      </div>

      {/* Product image */}
      <Link href={`/product/${slug}`} className="block relative">
        <div className="relative h-64 w-full overflow-hidden bg-gray-50 p-4">
          <div className="absolute inset-0 bg-gradient-to-br from-purple-50 to-white opacity-80"></div>
          <img
            src={image || "/placeholder.svg"}
            alt={name}
            className="w-full h-full object-contain relative z-10 transition-transform duration-500 group-hover:scale-110"
            onError={(e) => {
              e.target.onerror = null
              e.target.src = "/placeholder.svg?key=h4j3y"
            }}
          />
        </div>
      </Link>

      {/* Quick actions */}
      <div
        className={`absolute left-0 right-0 bottom-[5.5rem] flex justify-center gap-2 transition-all duration-300 ${isHovered ? "opacity-100 transform translate-y-0" : "opacity-0 transform translate-y-4"
          }`}
      >
        <button className="bg-white p-2 rounded-full shadow-md hover:bg-purple-600 hover:text-white transition-colors">
          <svg
            xmlns="http://www.w3.org/2000/svg"
            className="h-5 w-5"
            fill="none"
            viewBox="0 0 24 24"
            stroke="currentColor"
          >
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
            <path
              strokeLinecap="round"
              strokeLinejoin="round"
              strokeWidth={2}
              d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"
            />
          </svg>
        </button>
        <button className="bg-white p-2 rounded-full shadow-md hover:bg-purple-600 hover:text-white transition-colors">
          <svg
            xmlns="http://www.w3.org/2000/svg"
            className="h-5 w-5"
            fill="none"
            viewBox="0 0 24 24"
            stroke="currentColor"
          >
            <path
              strokeLinecap="round"
              strokeLinejoin="round"
              strokeWidth={2}
              d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"
            />
          </svg>
        </button>
        <button className="bg-white p-2 rounded-full shadow-md hover:bg-purple-600 hover:text-white transition-colors">
          <svg
            xmlns="http://www.w3.org/2000/svg"
            className="h-5 w-5"
            fill="none"
            viewBox="0 0 24 24"
            stroke="currentColor"
          >
            <path
              strokeLinecap="round"
              strokeLinejoin="round"
              strokeWidth={2}
              d="M8 7h12m0 0l-4-4m4 4l-4 4m-8 6H4m0 0l4 4m-4-4l4-4"
            />
          </svg>
        </button>
      </div>

      {/* Product details */}
      <div className="p-5">
        <Link href={`/product/${slug}`}>
          <h3 className="font-bold text-lg text-gray-900 hover:text-purple-600 transition-colors mb-1 line-clamp-1">
            {name}
          </h3>
        </Link>

        {/* Rating */}
        <div className="flex items-center mb-2">
          <div className="flex">
            {[...Array(5)].map((_, i) => (
              <svg
                key={i}
                className={`w-4 h-4 ${i < Math.floor(rating) ? "text-yellow-400" : "text-gray-300"}`}
                fill="currentColor"
                viewBox="0 0 20 20"
              >
                <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
              </svg>
            ))}
          </div>
          <span className="text-xs text-gray-500 ml-1">
            ({rating}) {reviewCount} reviews
          </span>
        </div>

        {/* Short description */}
        <p className="text-sm text-gray-600 mb-4 line-clamp-2">{shortDescription}</p>

        <div className="flex items-center justify-between">
          <div className="flex flex-col">
            {originalPrice && (
              <span className="text-xs text-gray-500 line-through">€{originalPrice.toFixed(2)}</span>
            )}
            <span className="text-xl font-bold text-gray-900">€{price.toFixed(2)}</span>
          </div>

          <button
            className={`px-4 py-2 rounded-lg transition-colors text-white font-medium ${stock === 0 ? "bg-gray-300 cursor-not-allowed" : "bg-purple-600 hover:bg-purple-700"
              }`}
            disabled={stock === 0}
          >
            {stock === 0 ? "Uitverkocht" : "In winkelwagen"}
          </button>
        </div>
      </div>
    </div>
  )
}

export default ProductCard
